import { DataTypes, Model,type Optional } from 'sequelize';
import sequelize from '../config/database.js';

export interface IServiceAttributes {
  id: number;
  title: string;
  iconKey: string;
  description: string;
  details: string | null;
  extraDescription: string[] | null;
  features: string[];
  included: string[] | null;
  notIncluded: string[] | null;
  terms: string[] | null;
  faqs: Array<{ q: string; a: string }> | null;
  createdAt: Date;
  updatedAt: Date;
}

export interface IServiceCreationAttributes extends Optional<IServiceAttributes, 'id' | 'details' | 'extraDescription' | 'included' | 'notIncluded' | 'terms' | 'faqs' | 'createdAt' | 'updatedAt'> {}

export interface IServiceInstance extends Model<IServiceAttributes, IServiceCreationAttributes>, IServiceAttributes {}

const Service = sequelize.define<IServiceInstance, IServiceCreationAttributes>('Service', {
  id: {
    type: DataTypes.INTEGER,
    primaryKey: true,
    autoIncrement: true
  },
  title: {
    type: DataTypes.STRING(100),
    allowNull: false,
    validate: {
      notEmpty: true,
      len: [1, 100]
    }
  },
  iconKey: {
    type: DataTypes.STRING,
    allowNull: false,
    validate: {
      notEmpty: true
    }
  },
  description: {
    type: DataTypes.STRING(500),
    allowNull: false,
    validate: {
      notEmpty: true,
      len: [1, 500]
    }
  },
  details: {
    type: DataTypes.STRING(1000),
    allowNull: true,
    validate: {
      len: [0, 1000]
    }
  },
  extraDescription: {
    type: DataTypes.JSON,
    allowNull: true,
    validate: {
      isValidArray(value: any) {
        if (value && !Array.isArray(value)) {
          throw new Error('extraDescription must be an array');
        }
        if (value) {
          value.forEach((item: any) => {
            if (typeof item !== 'string' || item.length > 500) {
              throw new Error('Each extraDescription item must be a string with max 500 characters');
            }
          });
        }
      }
    }
  },
  features: {
    type: DataTypes.JSON,
    allowNull: false,
    validate: {
      isValidFeaturesArray(value: any) {
        if (!Array.isArray(value) || value.length === 0) {
          throw new Error('features must be a non-empty array');
        }
        value.forEach((item: any) => {
          if (typeof item !== 'string' || item.length > 100) {
            throw new Error('Each feature must be a string with max 100 characters');
          }
        });
      }
    }
  },
  included: {
    type: DataTypes.JSON,
    allowNull: true,
    validate: {
      isValidArray(value: any) {
        if (value && !Array.isArray(value)) {
          throw new Error('included must be an array');
        }
        if (value) {
          value.forEach((item: any) => {
            if (typeof item !== 'string' || item.length > 100) {
              throw new Error('Each included item must be a string with max 100 characters');
            }
          });
        }
      }
    }
  },
  notIncluded: {
    type: DataTypes.JSON,
    allowNull: true,
    validate: {
      isValidArray(value: any) {
        if (value && !Array.isArray(value)) {
          throw new Error('notIncluded must be an array');
        }
        if (value) {
          value.forEach((item: any) => {
            if (typeof item !== 'string' || item.length > 100) {
              throw new Error('Each notIncluded item must be a string with max 100 characters');
            }
          });
        }
      }
    }
  },
  terms: {
    type: DataTypes.JSON,
    allowNull: true,
    validate: {
      isValidArray(value: any) {
        if (value && !Array.isArray(value)) {
          throw new Error('terms must be an array');
        }
        if (value) {
          value.forEach((item: any) => {
            if (typeof item !== 'string' || item.length > 200) {
              throw new Error('Each term must be a string with max 200 characters');
            }
          });
        }
      }
    }
  },
  faqs: {
    type: DataTypes.JSON,
    allowNull: true,
    validate: {
      isValidFaqsArray(value: any) {
        if (value && !Array.isArray(value)) {
          throw new Error('faqs must be an array');
        }
        if (value) {
          value.forEach((item: any) => {
            if (!item.q || !item.a || typeof item.q !== 'string' || typeof item.a !== 'string') {
              throw new Error('Each FAQ must have q and a properties as strings');
            }
            if (item.q.length > 200 || item.a.length > 500) {
              throw new Error('FAQ question max 200 chars, answer max 500 chars');
            }
          });
        }
      }
    }
  }
}, {
  tableName: 'services',
  timestamps: true
});

export default Service;