// import { Request, Response } from 'express';
import type { Request, Response } from 'express';
import WhatWeDo from '../models/WhatWeDo.js';

// Get all What We Do items
export const getWhatWeDoItems = async (req: Request, res: Response) => {
  try {
    const items = await WhatWeDo.findAll({
      where: { isActive: true },
      order: [['order', 'ASC']],
    });

    res.status(200).json({
      success: true,
      data: items,
    });
  } catch (error) {
    console.error('Error fetching What We Do items:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch What We Do items',
      error: error instanceof Error ? error.message : 'Unknown error',
    });
  }
};

// Get all What We Do items (admin)
export const getAllWhatWeDoItems = async (req: Request, res: Response) => {
  try {
    const items = await WhatWeDo.findAll({
      order: [['order', 'ASC']],
    });

    res.status(200).json({
      success: true,
      data: items,
    });
  } catch (error) {
    console.error('Error fetching all What We Do items:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch What We Do items',
      error: error instanceof Error ? error.message : 'Unknown error',
    });
  }
};

// Create new What We Do item
export const createWhatWeDoItem = async (req: Request, res: Response) => {
  try {
    const {
      icon,
      title,
      description,
      features,
      color,
      bgColor,
      iconColor,
      position,
      order,
      isActive = true,
    } = req.body;

    // Validate required fields
    if (!icon || !title || !description || !features || !color || !bgColor || !iconColor || !position) {
      return res.status(400).json({
        success: false,
        message: 'Missing required fields',
      });
    }

    const item = await WhatWeDo.create({
      icon,
      title,
      description,
      features,
      color,
      bgColor,
      iconColor,
      position,
      order: order || 0,
      isActive,
    });

    res.status(201).json({
      success: true,
      data: item,
      message: 'What We Do item created successfully',
    });
  } catch (error) {
    console.error('Error creating What We Do item:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to create What We Do item',
      error: error instanceof Error ? error.message : 'Unknown error',
    });
  }
};

// Update What We Do item
export const updateWhatWeDoItem = async (req: Request, res: Response) => {
  try {
    const { id } = req.params;
    const updateData = req.body;

    const item = await WhatWeDo.findByPk(id);
    if (!item) {
      return res.status(404).json({
        success: false,
        message: 'What We Do item not found',
      });
    }

    await item.update(updateData);

    res.status(200).json({
      success: true,
      data: item,
      message: 'What We Do item updated successfully',
    });
  } catch (error) {
    console.error('Error updating What We Do item:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update What We Do item',
      error: error instanceof Error ? error.message : 'Unknown error',
    });
  }
};

// Delete What We Do item
export const deleteWhatWeDoItem = async (req: Request, res: Response) => {
  try {
    const { id } = req.params;

    const item = await WhatWeDo.findByPk(id);
    if (!item) {
      return res.status(404).json({
        success: false,
        message: 'What We Do item not found',
      });
    }

    await item.destroy();

    res.status(200).json({
      success: true,
      message: 'What We Do item deleted successfully',
    });
  } catch (error) {
    console.error('Error deleting What We Do item:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to delete What We Do item',
      error: error instanceof Error ? error.message : 'Unknown error',
    });
  }
};

// Reorder What We Do items
export const reorderWhatWeDoItems = async (req: Request, res: Response) => {
  try {
    const { items } = req.body;

    if (!Array.isArray(items)) {
      return res.status(400).json({
        success: false,
        message: 'Items must be an array',
      });
    }

    // Update order for each item
    const updatePromises = items.map((item: { id: number; order: number }) =>
      WhatWeDo.update({ order: item.order }, { where: { id: item.id } })
    );

    await Promise.all(updatePromises);

    res.status(200).json({
      success: true,
      message: 'What We Do items reordered successfully',
    });
  } catch (error) {
    console.error('Error reordering What We Do items:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to reorder What We Do items',
      error: error instanceof Error ? error.message : 'Unknown error',
    });
  }
};
