import type { Request, Response } from "express";
import TeamMember from "../models/TeamMember.js";
import { toPublicUrl } from "../middleware/upload.js"; // ✅ helper we created

export const getAllTeamMembers = async (_req: Request, res: Response): Promise<void> => {
  try {
    const teamMembers = await TeamMember.findAll({ 
      order: [['createdAt', 'DESC']] 
    });
    res.json(teamMembers);
  } catch (error: any) {
    console.error("Get team members error:", error);
    res.status(500).json({ message: error.message || "Server error" });
  }
};

export const getTeamMemberById = async (req: Request, res: Response): Promise<void> => {
  try {
    const teamMember = await TeamMember.findByPk(req.params.id);
    if (!teamMember) {
      res.status(404).json({ message: "Team member not found" });
      return;
    }
    res.json(teamMember);
  } catch (error: any) {
    console.error("Get team member error:", error);
    res.status(500).json({ message: error.message || "Server error" });
  }
};

export const createTeamMember = async (req: Request, res: Response): Promise<void> => {
  try {
    const { name, role } = req.body;

    let imageUrl: string | undefined;
    if (req.file) {
      imageUrl = toPublicUrl(req.file.filename);
    }

    const teamMember = await TeamMember.create({
      name,
      role,
      image: imageUrl || '',
    });
    res.status(201).json(teamMember);
  } catch (error: any) {
    console.error("Create team member error:", error);
    res.status(400).json({ message: error.message || "Server error" });
  }
};

export const updateTeamMember = async (req: Request, res: Response): Promise<void> => {
  try {
    const { name, role } = req.body;

    const updateData: any = { name, role };
    if (req.file) {
      updateData.image = toPublicUrl(req.file.filename);
    }

    const [updatedRows] = await TeamMember.update(updateData, {
      where: { id: req.params.id },
      returning: true
    });

    if (updatedRows === 0) {
      res.status(404).json({ message: "Team member not found" });
      return;
    }

    const teamMember = await TeamMember.findByPk(req.params.id);
    res.json(teamMember);
  } catch (error: any) {
    console.error("Update team member error:", error);
    res.status(400).json({ message: error.message || "Server error" });
  }
};

export const deleteTeamMember = async (req: Request, res: Response): Promise<void> => {
  try {
    const deletedRows = await TeamMember.destroy({
      where: { id: req.params.id }
    });
    
    if (deletedRows === 0) {
      res.status(404).json({ message: "Team member not found" });
      return;
    }
    
    res.json({ message: "Team member deleted successfully" });
  } catch (error: any) {
    console.error("Delete team member error:", error);
    res.status(500).json({ message: error.message || "Server error" });
  }
};
  