import type { Request, Response } from 'express';
import Config from '../models/Config.js';

// Store OTPs in memory (in production, use Redis or database)
export const otpStore = new Map<string, { otp: string; expiresAt: number; purpose: string }>();

export const sendOtp = async (req: Request, res: Response): Promise<void> => {
  try {
    const { phone, purpose } = req.body;

    if (!phone || !purpose) {
      res.status(400).json({ message: 'Phone and purpose are required' });
      return;
    }

    // Format phone number to ensure consistency (add +91 if not present)
    let formattedPhone = phone;
    if (!phone.startsWith('+91') && !phone.startsWith('91')) {
      formattedPhone = phone.startsWith('+') ? phone : `+91${phone}`;
    } else if (phone.startsWith('91') && !phone.startsWith('+91')) {
      formattedPhone = `${phone}`;
    }

    // Generate 6-digit OTP
    const otp = Math.floor(100000 + Math.random() * 900000).toString();
    const expiresAt = Date.now() + 10 * 60 * 1000; // 10 minutes

    // Store OTP with formatted phone number
    otpStore.set(formattedPhone, { otp, expiresAt, purpose });

    // Get WhatsApp credentials from config
    const whatsappConfig = await Config.findOne({
      where: { key: 'WHATSAPP_INSTANCE_ID' }
    });
    const whatsappTokenConfig = await Config.findOne({
      where: { key: 'WHATSAPP_ACCESS_TOKEN' }
    });

    if (!whatsappConfig || !whatsappTokenConfig) {
      console.log('WhatsApp credentials not found in config, OTP:', otp);
      res.json({ message: 'OTP generated (WhatsApp not configured)', otp: otp });
      return;
    }

    const instanceId = whatsappConfig.value;
    const accessToken = whatsappTokenConfig.value;
console.log("formattedPhone",formattedPhone);

    // Send WhatsApp message using Msg91 API
    try {
      const message = `Your OTP for ${purpose} is: ${otp}. Valid for 10 minutes.`;
      const whatsappUrl = `https://msg91.in/api/send?number=${formattedPhone}&type=text&message=${encodeURIComponent(message)}&instance_id=${instanceId}&access_token=${accessToken}`;
      
      const whatsappResponse = await fetch(whatsappUrl, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
        }
      });

      if (whatsappResponse.ok) {
        console.log('WhatsApp OTP sent successfully');
        res.json({ message: 'OTP sent via WhatsApp' });
      } else {
        console.error('WhatsApp API error:', await whatsappResponse.text());
        res.json({ message: 'OTP generated (WhatsApp failed)', otp: otp });
      }
    } catch (whatsappError) {
      console.error('WhatsApp error:', whatsappError);
      res.json({ message: 'OTP generated (WhatsApp error)', otp: otp });
    }

  } catch (error: any) {
    console.error('Send OTP error:', error);
    res.status(500).json({ message: error.message || 'Server error' });
  }
};

export const verifyOtp = async (req: Request, res: Response): Promise<void> => {
  try {
    const { phone, otp, purpose } = req.body;

    if (!phone || !otp || !purpose) {
      res.status(400).json({ message: 'Phone, OTP, and purpose are required' });
      return;
    }

    // Format phone number to ensure consistency (add +91 if not present)
    let formattedPhone = phone;
    if (!phone.startsWith('+91') && !phone.startsWith('91')) {
      formattedPhone = phone.startsWith('+') ? phone : `+91${phone}`;
    } else if (phone.startsWith('91') && !phone.startsWith('+91')) {
      formattedPhone = `+${phone}`;
    }

    const storedOtp = otpStore.get(formattedPhone);

    if (!storedOtp) {
      // Try with original phone number as fallback
      const fallbackOtp = otpStore.get(phone);
      if (!fallbackOtp) {
        res.status(400).json({ message: 'OTP not found or expired' });
        return;
      }
      // Use fallback OTP
      if (fallbackOtp.otp !== otp) {
        res.status(400).json({ message: 'Invalid OTP' });
        return;
      }
      if (fallbackOtp.expiresAt < Date.now()) {
        res.status(400).json({ message: 'OTP has expired' });
        return;
      }
      // OTP is valid, remove it
      otpStore.delete(phone);
      res.json({ message: 'OTP verified successfully' });
      return;
    }

    if (storedOtp.otp !== otp) {
      res.status(400).json({ message: 'Invalid OTP' });
      return;
    }

    if (storedOtp.expiresAt < Date.now()) {
      otpStore.delete(formattedPhone);
      res.status(400).json({ message: 'OTP expired' });
      return;
    }

    if (storedOtp.purpose !== purpose) {
      res.status(400).json({ message: 'Invalid purpose' });
      return;
    }

    // Remove OTP after successful verification
    otpStore.delete(formattedPhone);

    res.json({ message: 'OTP verified successfully' });

  } catch (error: any) {
    console.error('Verify OTP error:', error);
    res.status(500).json({ message: error.message || 'Server error' });
  }
};
