// import { Request, Response } from 'express';
import type { Request, Response } from 'express';
import Career from '../models/Career.js';

// Get all job openings
export const getAllCareers = async (req: Request, res: Response) => {
  try {
    const { isActive, type, department } = req.query;
    
    let whereClause: any = {};
    
    if (isActive !== undefined) {
      whereClause.isActive = isActive === 'true';
    }
    
    if (type) {
      whereClause.type = type;
    }
    
    if (department) {
      whereClause.department = department;
    }
    
    const careers = await Career.findAll({
      where: whereClause,
      order: [['createdAt', 'DESC']]
    });
    
    res.json(careers);
  } catch (error) {
    console.error('Error fetching careers:', error);
    res.status(500).json({ message: 'Error fetching job openings' });
  }
};

// Get single job opening by ID
export const getCareerById = async (req: Request, res: Response) => {
  try {
    const { id } = req.params;
    const career = await Career.findByPk(id);
    
    if (!career) {
      return res.status(404).json({ message: 'Job opening not found' });
    }
    
    res.json(career);
  } catch (error) {
    console.error('Error fetching career:', error);
    res.status(500).json({ message: 'Error fetching job opening' });
  }
};

// Create new job opening
export const createCareer = async (req: Request, res: Response) => {
  try {
    const {
      title,
      location,
      type,
      experience,
      description,
      requirements,
      responsibilities,
      benefits,
      skills,
      salaryRange,
      department,
      isActive,
      applicationDeadline
    } = req.body;
    
    // Validate required fields
    if (!title || !location || !type || !experience || !description || !department) {
      return res.status(400).json({ 
        message: 'Missing required fields: title, location, type, experience, description, department' 
      });
    }
    
    const career = await Career.create({
      title,
      location,
      type,
      experience,
      description,
      requirements: requirements || [],
      responsibilities: responsibilities || [],
      benefits: benefits || [],
      skills: skills || [],
      salaryRange,
      department,
      isActive: isActive !== undefined ? isActive : true,
      applicationDeadline
    });
    
    res.status(201).json(career);
  } catch (error) {
    console.error('Error creating career:', error);
    res.status(500).json({ message: 'Error creating job opening' });
  }
};

// Update job opening
export const updateCareer = async (req: Request, res: Response) => {
  try {
    const { id } = req.params;
    const updateData = req.body;
    
    console.log('Update career request:', { id, updateData });
    
    const career = await Career.findByPk(id);
    
    if (!career) {
      console.log('Career not found with id:', id);
      return res.status(404).json({ message: 'Job opening not found' });
    }
    
    console.log('Found career:', career.toJSON());
    
    const updatedCareer = await career.update(updateData);
    
    console.log('Updated career:', updatedCareer.toJSON());
    
    res.json(updatedCareer);
  } catch (error) {
    console.error('Error updating career:', error);
    res.status(500).json({ message: 'Error updating job opening' });
  }
};

// Delete job opening
export const deleteCareer = async (req: Request, res: Response) => {
  try {
    const { id } = req.params;
    
    const career = await Career.findByPk(id);
    
    if (!career) {
      return res.status(404).json({ message: 'Job opening not found' });
    }
    
    await career.destroy();
    
    res.json({ message: 'Job opening deleted successfully' });
  } catch (error) {
    console.error('Error deleting career:', error);
    res.status(500).json({ message: 'Error deleting job opening' });
  }
};

// Toggle job opening status (active/inactive)
export const toggleCareerStatus = async (req: Request, res: Response) => {
  try {
    const { id } = req.params;
    
    const career = await Career.findByPk(id);
    
    if (!career) {
      return res.status(404).json({ message: 'Job opening not found' });
    }
    
    await career.update({ isActive: !career.isActive });
    
    res.json(career);
  } catch (error) {
    console.error('Error toggling career status:', error);
    res.status(500).json({ message: 'Error toggling job opening status' });
  }
};

// Get career statistics
export const getCareerStats = async (req: Request, res: Response) => {
  try {
    const totalCareers = await Career.count();
    const activeCareers = await Career.count({ where: { isActive: true } });
    const inactiveCareers = await Career.count({ where: { isActive: false } });
    
    // Count by type
    const careersByType = await Career.findAll({
      attributes: [
        'type',
        [Career.sequelize!.fn('COUNT', Career.sequelize!.col('id')), 'count']
      ],
      group: ['type'],
      raw: true
    });
    
    // Count by department
    const careersByDepartment = await Career.findAll({
      attributes: [
        'department',
        [Career.sequelize!.fn('COUNT', Career.sequelize!.col('id')), 'count']
      ],
      group: ['department'],
      raw: true
    });
    
    res.json({
      total: totalCareers,
      active: activeCareers,
      inactive: inactiveCareers,
      byType: careersByType,
      byDepartment: careersByDepartment
    });
  } catch (error) {
    console.error('Error fetching career stats:', error);
    res.status(500).json({ message: 'Error fetching career statistics' });
  }
};
