import type { Request, Response } from 'express';
import jwt from 'jsonwebtoken';
import User, { type IUserInstance } from '../models/User.js';

const generateToken = (userId: number): string => {
  return jwt.sign(
    { userId: userId.toString() },
    process.env.JWT_SECRET || 'your-secret-key',
    { expiresIn: '7d' }
  );
};


export const register = async (req: Request, res: Response): Promise<void> => {
  try {
    const { name, email, password } = req.body;

    const existingUser = await User.findOne({ where: { email } });
    if (existingUser) {
      res.status(400).json({ message: 'User already exists with this email' });
      return;
    }

    // Create new user
    const user = await User.create({ name, email, password });

    // Generate token
    const token = generateToken(user.id);

    res.status(201).json({
      message: 'User created successfully',
      user: {
        id: user.id,
        name: user.name,
        email: user.email,
      },
      token
    });
  } catch (error: any) {
    console.error('Registration error:', error);
    res.status(500).json({ message: error.message || 'Server error' });
  }
};

export const login = async (req: Request, res: Response): Promise<void> => {
  try {
    const { email, password } = req.body;

    // Find user by email
    const user = await User.findOne({ where: { email } });
    if (!user) {
      res.status(401).json({ message: 'Invalid credentials' });
      return;
    }

    // Check password
    const isPasswordValid = await user.comparePassword(password);
    if (!isPasswordValid) {
      res.status(401).json({ message: 'Invalid credentials' });
      return;
    }

    // Generate token
    const token = generateToken(user.id);

    res.json({
      message: 'Login successful',
      user: {
        id: user.id,
        name: user.name,
        email: user.email,
        role: user.role
      },
      token
    });
  } catch (error: any) {
    console.error('Login error:', error);
    res.status(500).json({ message: error.message || 'Server error' });
  }
};


export const getCurrentUser = async (req: Request, res: Response): Promise<void> => {
  try {
    const user = req.user as IUserInstance;
    res.json({
      id: user.id,
      name: user.name,
      email: user.email,
      role: user.role
    });
  } catch (error: any) {
    console.error('Get current user error:', error);
    res.status(500).json({ message: error.message || 'Server error' });
  }
}

export const logout = async (req: Request, res: Response): Promise<void> => {
  res.json({ message: 'Logout successful' });
};

export const forgotPassword = async (req: Request, res: Response) => {
  try {
    const { email } = req.body;
    const user = await User.findOne({ where: { email } });
    if (!user) {
      return res.status(404).json({ message: "User not found" });
    }

    // Generate 6-digit OTP
    const otp = Math.floor(100000 + Math.random() * 900000).toString();

    await user.update({
      resetOtp: otp,
      resetOtpExpiry: Date.now() + 10 * 60 * 1000 // valid for 10 min
    });

    // Send email (setup nodemailer transport first)
    // const transporter = nodemailer.createTransport({
    //   service: "gmail", // or use SMTP
    //   auth: {
    //     user: process.env.EMAIL_USER,
    //     pass: process.env.EMAIL_PASS,
    //   },
    // });

    // await transporter.sendMail({
    //   from: `"Support" <${process.env.EMAIL_USER}>`,
    //   to: user.email,
    //   subject: "Password Reset OTP",
    //   text: `Your OTP code is ${otp}. It is valid for 10 minutes.`,
    // });

    res.json({ message: "OTP sent to email" });
  } catch (error: any) {
    console.error("Forgot Password error:", error);
    res.status(500).json({ message: error.message || "Server error" });
  }
};

// STEP 2: Verify OTP
export const verifyOtp = async (req: Request, res: Response) => {
  try {
    const { email, otp } = req.body;
    const user = await User.findOne({ where: { email } });
    if (
      !user ||
      user.resetOtp !== otp ||
      !user.resetOtpExpiry ||
      user.resetOtpExpiry < Date.now()
    ) {
      return res.status(400).json({ message: "Invalid or expired OTP" });
    }

    res.json({ message: "OTP verified successfully" });
  } catch (error: any) {
    console.error("Verify OTP error:", error);
    res.status(500).json({ message: error.message || "Server error" });
  }
};

// STEP 3: Reset Password
export const resetPassword = async (req: Request, res: Response) => {
  try {
    const { email, password, confirmPassword } = req.body;

    if (password !== confirmPassword) {
      return res.status(400).json({ message: "Passwords do not match" });
    }

    const user = await User.findOne({ where: { email } });
    if (!user) return res.status(404).json({ message: "User not found" });

    await user.update({
      password: password,
      resetOtp: null,    
      resetOtpExpiry: null
    });

    res.json({ message: "Password reset successful" });
  } catch (error: any) {
    console.error("Reset Password error:", error);
    res.status(500).json({ message: error.message || "Server error" });
  }
};