// Temporary storage service for job application data
// In production, this should use Redis with expiration

interface TempApplicationData {
  careerId: number;
  name: string;
  email: string;
  mobile: string;
  hasExperience: boolean;
  experienceDetails?: string;
  coverLetter?: string | null;
  resume?: string | null;
  timestamp: number;
}

class TempStorageService {
  private storage = new Map<string, TempApplicationData>();

  // Store application data temporarily
  store(mobile: string, data: TempApplicationData): void {
    this.storage.set(mobile, data);
  }

  // Get application data
  get(mobile: string): TempApplicationData | undefined {
    return this.storage.get(mobile);
  }

  // Remove application data
  remove(mobile: string): boolean {
    return this.storage.delete(mobile);
  }

  // Check if data exists and is not expired
  isValid(mobile: string, maxAgeMinutes: number = 30): boolean {
    const data = this.storage.get(mobile);
    if (!data) return false;
    
    const maxAge = maxAgeMinutes * 60 * 1000; // Convert to milliseconds
    return (Date.now() - data.timestamp) <= maxAge;
  }

  // Clean up expired data
  cleanup(maxAgeMinutes: number = 30): void {
    const maxAge = maxAgeMinutes * 60 * 1000;
    const now = Date.now();
    
    for (const [mobile, data] of this.storage.entries()) {
      if (now - data.timestamp > maxAge) {
        this.storage.delete(mobile);
      }
    }
  }

  // Get all stored data (for debugging)
  getAll(): Map<string, TempApplicationData> {
    return new Map(this.storage);
  }
}

// Export singleton instance
export const tempStorage = new TempStorageService();

// Clean up expired data every 10 minutes
setInterval(() => {
  tempStorage.cleanup();
}, 10 * 60 * 1000);
