import sequelize from './config/database.js';
import User from './models/User.js';
import Blog from './models/Blog.js';
import Service from './models/Service.js';
import TeamMember from './models/TeamMember.js';
import Testimonial from './models/Testimonial.js';
import Feature from './models/Feature.js';
import Notification from './models/Notification.js';
import Banner from './models/Banner.js';
import Company from './models/Company.js';
import Career from './models/Career.js';

const seedDatabase = async () => {
  try {
    console.log('🌱 Starting database seeding...');

    // Create admin user (check if exists first)
    let adminUser = await User.findOne({ where: { email: 'admin@gmail.com' } });
    if (!adminUser) {
      adminUser = await User.create({
        name: 'Admin User',
        email: 'admin@gmail.com',
        password: '121212',
        role: 'admin'
      });
      console.log('✅ Admin user created:', adminUser.email);
    } else {
      console.log('ℹ️ Admin user already exists:', adminUser.email);
    }

    // Create sample blogs (check if any exist first)
    const existingBlogs = await Blog.count();
    let blogs = [];
    if (existingBlogs === 0) {
      blogs = await Blog.bulkCreate([
      {
        title: 'Getting Started with Web Development',
        category: 'Technology',
        author: 'John Doe',
        date: '2024-01-15',
        comments: 5,
        rating: 4.5,
        desc: 'Learn the fundamentals of web development with this comprehensive guide.',
        href: '/blog/web-development-guide',
        image: '/uploads/sample-blog-1.jpg',
        content: 'Web development is an exciting field that combines creativity with technical skills...'
      },
      {
        title: 'Best Practices for React Development',
        category: 'Programming',
        author: 'Jane Smith',
        date: '2024-01-20',
        comments: 12,
        rating: 4.8,
        desc: 'Discover the best practices for building scalable React applications.',
        href: '/blog/react-best-practices',
        image: '/uploads/sample-blog-2.jpg',
        content: 'React has become one of the most popular JavaScript libraries for building user interfaces...'
      },
      {
        title: 'Database Design Fundamentals',
        category: 'Database',
        author: 'Mike Johnson',
        date: '2024-01-25',
        comments: 8,
        rating: 4.3,
        desc: 'Master the art of database design with these essential principles.',
        href: '/blog/database-design',
        image: '/uploads/sample-blog-3.jpg',
        content: 'A well-designed database is the foundation of any successful application...'
      }
    ]);
      console.log('✅ Sample blogs created:', blogs.length);
    } else {
      console.log('ℹ️ Blogs already exist, skipping blog creation');
    }

    // Create sample services (check if any exist first)
    const existingServices = await Service.count();
    let services = [];
    if (existingServices === 0) {
      services = await Service.bulkCreate([
      {
        title: 'Web Development',
        iconKey: 'code',
        description: 'Custom web applications built with modern technologies',
        details: 'We create responsive, scalable web applications using the latest frameworks and technologies.',
        extraDescription: [
          'Responsive design for all devices',
          'Modern JavaScript frameworks',
          'Database integration',
          'API development'
        ],
        features: ['Frontend Development', 'Backend Development', 'Database Design', 'API Integration'],
        included: ['Source Code', 'Documentation', 'Testing', 'Deployment'],
        notIncluded: ['Domain Registration', 'Hosting Setup', 'Third-party Services'],
        terms: ['Project timeline: 4-8 weeks', 'Payment: 50% upfront, 50% on completion'],
        faqs: [
          { q: 'What technologies do you use?', a: 'We use React, Node.js, and modern databases like MySQL and MongoDB.' },
          { q: 'Do you provide maintenance?', a: 'Yes, we offer ongoing maintenance and support packages.' }
        ]
      },
      {
        title: 'Mobile App Development',
        iconKey: 'smartphone',
        description: 'Native and cross-platform mobile applications',
        details: 'We develop mobile apps for iOS and Android using native and cross-platform solutions.',
        extraDescription: [
          'iOS and Android development',
          'Cross-platform solutions',
          'App store optimization',
          'Push notifications'
        ],
        features: ['iOS Development', 'Android Development', 'Cross-platform', 'App Store Deployment'],
        included: ['App Development', 'Testing', 'App Store Submission', 'Basic Support'],
        notIncluded: ['App Store Fees', 'Third-party Integrations', 'Advanced Analytics'],
        terms: ['Project timeline: 6-12 weeks', 'Payment: 30% upfront, 40% mid-project, 30% on completion'],
        faqs: [
          { q: 'Do you develop for both platforms?', a: 'Yes, we develop for both iOS and Android platforms.' },
          { q: 'What about app store approval?', a: 'We handle the entire app store submission process.' }
        ]
      }
    ]);
      console.log('✅ Sample services created:', services.length);
    } else {
      console.log('ℹ️ Services already exist, skipping service creation');
    }

    // Create sample team members (check if any exist first)
    const existingTeamMembers = await TeamMember.count();
    let teamMembers = [];
    if (existingTeamMembers === 0) {
      teamMembers = await TeamMember.bulkCreate([
      {
        name: 'John Doe',
        role: 'Senior Developer',
        image: '/uploads/team-member-1.jpg'
      },
      {
        name: 'Jane Smith',
        role: 'UI/UX Designer',
        image: '/uploads/team-member-2.jpg'
      },
      {
        name: 'Mike Johnson',
        role: 'Project Manager',
        image: '/uploads/team-member-3.jpg'
      },
      {
        name: 'Sarah Wilson',
        role: 'Backend Developer',
        image: '/uploads/team-member-4.jpg'
      }
    ]);
      console.log('✅ Sample team members created:', teamMembers.length);
    } else {
      console.log('ℹ️ Team members already exist, skipping team member creation');
    }

    // Create sample testimonials (check if any exist first)
    const existingTestimonials = await Testimonial.count();
    let testimonials = [];
    if (existingTestimonials === 0) {
      testimonials = await Testimonial.bulkCreate([
      {
        name: 'Alice Brown',
        role: 'CEO, TechCorp',
        text: 'Inficom Solutions delivered an exceptional web application that exceeded our expectations. Their team was professional and responsive throughout the project.',
        image: '/uploads/testimonial-1.jpg',
        stars: 5,
        size: 'large'
      },
      {
        name: 'Bob Davis',
        role: 'Founder, StartupXYZ',
        text: 'The mobile app they developed for us has been a game-changer. User engagement increased by 300% after launch.',
        image: '/uploads/testimonial-2.jpg',
        stars: 5,
        size: 'small'
      },
      {
        name: 'Carol Green',
        role: 'CTO, Enterprise Inc',
        text: 'Outstanding technical expertise and attention to detail. They delivered on time and within budget.',
        image: '/uploads/testimonial-3.jpg',
        stars: 4,
        size: 'small'
      },
      {
        name: 'David Lee',
        role: 'Product Manager, InnovateCo',
        text: 'Working with Inficom Solutions was a pleasure. They understood our vision and brought it to life perfectly.',
        image: '/uploads/testimonial-4.jpg',
        stars: 5,
        size: 'large'
      }
    ]);
      console.log('✅ Sample testimonials created:', testimonials.length);
    } else {
      console.log('ℹ️ Testimonials already exist, skipping testimonial creation');
    }

    // Create sample features (check if any exist first)
    const existingFeatures = await Feature.count();
    let features = [];
    if (existingFeatures === 0) {
      features = await Feature.bulkCreate([
      {
        title: 'Responsive Design',
        desc: 'Our applications work seamlessly across all devices and screen sizes',
        iconKey: 'responsive',
        color: '#3B82F6'
      },
      {
        title: 'Fast Performance',
        desc: 'Optimized for speed and performance to provide the best user experience',
        iconKey: 'speed',
        color: '#10B981'
      },
      {
        title: 'Secure & Reliable',
        desc: 'Built with security best practices and reliable infrastructure',
        iconKey: 'security',
        color: '#F59E0B'
      },
      {
        title: 'Scalable Solutions',
        desc: 'Architecture designed to grow with your business needs',
        iconKey: 'scale',
        color: '#8B5CF6'
      },
      {
        title: '24/7 Support',
        desc: 'Round-the-clock support to ensure your applications run smoothly',
        iconKey: 'support',
        color: '#EF4444'
      },
      {
        title: 'Modern Technology',
        desc: 'Using the latest technologies and frameworks for future-proof solutions',
        iconKey: 'technology',
        color: '#06B6D4'
      }
    ]);
      console.log('✅ Sample features created:', features.length);
    } else {
      console.log('ℹ️ Features already exist, skipping feature creation');
    }

    // Create sample notifications (check if any exist first)
    const existingNotifications = await Notification.count();
    let notifications = [];
    if (existingNotifications === 0) {
      notifications = await Notification.bulkCreate([
      {
        notificationId: 'welcome-notification',
        message: 'Welcome to Inficom Solutions! We are excited to work with you.',
        link: '/welcome',
        color: '#3B82F6',
        isActive: true
      },
      {
        notificationId: 'maintenance-notice',
        message: 'Scheduled maintenance will occur on Sunday from 2-4 AM EST.',
        link: '/maintenance',
        color: '#F59E0B',
        isActive: false
      },
      {
        notificationId: 'new-feature',
        message: 'Check out our new dashboard features and improved user interface!',
        link: '/features',
        color: '#10B981',
        isActive: false
      }
    ]);
      console.log('✅ Sample notifications created:', notifications.length);
    } else {
      console.log('ℹ️ Notifications already exist, skipping notification creation');
    }

    // Create sample banners (check if any exist first)
    const existingBanners = await Banner.count();
    let banners = [];
    if (existingBanners === 0) {
      banners = await Banner.bulkCreate([
        {
          message: 'Welcome to Inficom Solutions! We are excited to work with you.',
          link: '/welcome',
          color: '#3B82F6',
          isActive: true
        },
        {
          message: 'New features available! Check out our latest updates.',
          link: '/features',
          color: '#10B981',
          isActive: false
        },
        {
          message: 'Special offer: 20% off on all web development services this month!',
          link: '/offers',
          color: '#F59E0B',
          isActive: false
        }
      ]);
      console.log('✅ Sample banners created:', banners.length);
    } else {
      console.log('ℹ️ Banners already exist, skipping banner creation');
    }

    // Create sample company (check if exists first)
    const existingCompany = await Company.count();
    let company = null;
    if (existingCompany === 0) {
      company = await Company.create({
        name: 'Inficom Solutions',
        email: 'info@inficomsolutions.com',
        phone: '+1 (555) 123-4567',
        address: '123 Tech Street, Innovation District, City, State 12345',
        description: 'We are a leading technology company specializing in web development, mobile applications, and digital solutions. Our team of experienced developers and designers work together to deliver exceptional results for our clients.',
        website: 'https://inficomsolutions.com',
        socialMedia: {
          facebook: 'https://facebook.com/inficomsolutions',
          twitter: 'https://twitter.com/inficomsolutions',
          linkedin: 'https://linkedin.com/company/inficomsolutions',
          instagram: 'https://instagram.com/inficomsolutions'
        }
      });
      console.log('✅ Sample company created:', company.name);
    } else {
      console.log('ℹ️ Company already exists, skipping company creation');
    }

    // Create sample careers (check if any exist first)
    const existingCareers = await Career.count();
    let careers = [];
    if (existingCareers === 0) {
      careers = await Career.bulkCreate([
        {
          title: "MERN Full Stack Developer",
          location: "Aurangabad, Maharashtra",
          type: "Full-time",
          experience: "2-4 years",
          description: "We're looking for a skilled MERN stack developer to join our development team. You'll work on building scalable web applications using MongoDB, Express.js, React, and Node.js. This role involves both frontend and backend development, working closely with our design and product teams.",
          requirements: [
            "2+ years of experience with MERN stack development",
            "Proficiency in JavaScript, ES6+, and TypeScript",
            "Experience with MongoDB and database design",
            "Strong knowledge of React.js and state management (Redux/Context)",
            "Experience with Node.js and Express.js",
            "Knowledge of RESTful APIs and GraphQL",
            "Experience with version control (Git)",
            "Understanding of responsive design principles"
          ],
          responsibilities: [
            "Develop and maintain web applications using MERN stack",
            "Design and implement RESTful APIs",
            "Collaborate with frontend and backend teams",
            "Write clean, maintainable, and efficient code",
            "Participate in code reviews and technical discussions",
            "Troubleshoot and debug applications",
            "Stay updated with latest technologies and best practices"
          ],
          benefits: [
            "Competitive salary package",
            "Health insurance coverage",
            "Flexible working hours",
            "Remote work options",
            "Professional development opportunities",
            "Team building activities",
            "Performance bonuses"
          ],
          skills: ["JavaScript", "React", "Node.js", "Express.js", "MongoDB", "TypeScript", "Git", "RESTful APIs"],
          salaryRange: "₹4,00,000 - ₹7,00,000",
          department: "Engineering",
          isActive: true,
          applicationDeadline: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000)
        },
        {
          title: "Flutter Developer",
          location: "Aurangabad, Maharashtra",
          type: "Full-time",
          experience: "1-3 years",
          description: "Join our mobile development team as a Flutter developer. You'll be responsible for developing cross-platform mobile applications for iOS and Android using Flutter framework. This role offers the opportunity to work on innovative mobile solutions and contribute to our growing mobile app portfolio.",
          requirements: [
            "1+ years of experience with Flutter development",
            "Proficiency in Dart programming language",
            "Experience with mobile app development lifecycle",
            "Knowledge of iOS and Android platform guidelines",
            "Experience with state management (Provider, Bloc, Riverpod)",
            "Understanding of RESTful APIs and JSON",
            "Experience with version control (Git)",
            "Portfolio showcasing Flutter applications"
          ],
          responsibilities: [
            "Develop cross-platform mobile applications using Flutter",
            "Collaborate with UI/UX designers to implement designs",
            "Integrate with backend APIs and services",
            "Write unit and integration tests",
            "Optimize app performance and user experience",
            "Maintain and update existing mobile applications",
            "Stay updated with Flutter and mobile development trends"
          ],
          benefits: [
            "Competitive salary package",
            "Health insurance coverage",
            "Flexible working hours",
            "Remote work options",
            "Learning and development budget",
            "Latest development tools and equipment",
            "Team outings and events"
          ],
          skills: ["Flutter", "Dart", "Mobile Development", "iOS", "Android", "RESTful APIs", "Git", "State Management"],
          salaryRange: "₹3,50,000 - ₹6,00,000",
          department: "Engineering",
          isActive: true,
          applicationDeadline: new Date(Date.now() + 25 * 24 * 60 * 60 * 1000)
        },
        {
          title: "Frontend Developer",
          location: "Aurangabad, Maharashtra",
          type: "Full-time",
          experience: "1-3 years",
          description: "We're seeking a talented Frontend Developer to join our team. You'll be responsible for creating engaging user interfaces and experiences using modern web technologies. This role involves working closely with designers and backend developers to deliver high-quality web applications.",
          requirements: [
            "1+ years of frontend development experience",
            "Proficiency in HTML5, CSS3, and JavaScript",
            "Experience with React.js or Vue.js",
            "Knowledge of responsive web design",
            "Experience with CSS frameworks (Tailwind CSS, Bootstrap)",
            "Understanding of version control (Git)",
            "Basic knowledge of RESTful APIs",
            "Attention to detail and design aesthetics"
          ],
          responsibilities: [
            "Develop responsive web applications",
            "Implement UI/UX designs with pixel-perfect accuracy",
            "Optimize applications for maximum speed and scalability",
            "Collaborate with backend developers for API integration",
            "Ensure cross-browser compatibility",
            "Write clean, maintainable code",
            "Participate in design reviews and technical discussions"
          ],
          benefits: [
            "Competitive salary package",
            "Health insurance coverage",
            "Flexible working hours",
            "Remote work options",
            "Professional development opportunities",
            "Creative and collaborative work environment",
            "Performance-based bonuses"
          ],
          skills: ["HTML5", "CSS3", "JavaScript", "React", "Vue.js", "Tailwind CSS", "Git", "Responsive Design"],
          salaryRange: "₹3,00,000 - ₹5,50,000",
          department: "Engineering",
          isActive: true,
          applicationDeadline: new Date(Date.now() + 20 * 24 * 60 * 60 * 1000)
        },
        {
          title: "Graphics Designer",
          location: "Aurangabad, Maharashtra",
          type: "Full-time",
          experience: "1-3 years",
          description: "Join our creative team as a Graphics Designer. You'll be responsible for creating visually appealing designs for various marketing materials, web graphics, and branding elements. This role offers the opportunity to work on diverse projects and contribute to our company's visual identity.",
          requirements: [
            "1+ years of graphic design experience",
            "Proficiency in Adobe Creative Suite (Photoshop, Illustrator, InDesign)",
            "Knowledge of design principles and typography",
            "Experience with digital and print design",
            "Understanding of brand guidelines and visual identity",
            "Portfolio showcasing design skills",
            "Basic knowledge of web design principles",
            "Creative thinking and problem-solving skills"
          ],
          responsibilities: [
            "Create marketing materials and promotional graphics",
            "Design web graphics and user interface elements",
            "Develop brand identity and visual guidelines",
            "Collaborate with marketing and development teams",
            "Maintain brand consistency across all materials",
            "Create social media graphics and content",
            "Stay updated with design trends and tools"
          ],
          benefits: [
            "Competitive salary package",
            "Health insurance coverage",
            "Flexible working hours",
            "Remote work options",
            "Creative freedom and autonomy",
            "Latest design software and tools",
            "Team building and creative workshops"
          ],
          skills: ["Adobe Photoshop", "Adobe Illustrator", "Adobe InDesign", "Graphic Design", "Branding", "Typography", "Web Design", "Creative Suite"],
          salaryRange: "₹2,50,000 - ₹4,50,000",
          department: "Design",
          isActive: true,
          applicationDeadline: new Date(Date.now() + 35 * 24 * 60 * 60 * 1000)
        },
        {
          title: "Senior MERN Developer",
          location: "Aurangabad, Maharashtra",
          type: "Full-time",
          experience: "4-6 years",
          description: "We're looking for an experienced Senior MERN Developer to lead our development team. You'll be responsible for architecting scalable applications, mentoring junior developers, and driving technical decisions. This role offers leadership opportunities and the chance to work on complex, high-impact projects.",
          requirements: [
            "4+ years of MERN stack development experience",
            "Expert-level proficiency in JavaScript, TypeScript",
            "Deep understanding of React.js ecosystem and best practices",
            "Strong experience with Node.js and Express.js",
            "Advanced MongoDB knowledge and database optimization",
            "Experience with microservices architecture",
            "Leadership and mentoring experience",
            "Strong problem-solving and communication skills"
          ],
          responsibilities: [
            "Lead development of complex web applications",
            "Mentor and guide junior developers",
            "Architect scalable and maintainable solutions",
            "Make technical decisions and set coding standards",
            "Collaborate with product and design teams",
            "Conduct code reviews and ensure quality",
            "Stay updated with latest technologies and trends"
          ],
          benefits: [
            "Competitive salary package with equity options",
            "Comprehensive health insurance",
            "Flexible working hours and remote work",
            "Professional development budget",
            "Leadership development opportunities",
            "Performance bonuses and profit sharing",
            "Team leadership and management training"
          ],
          skills: ["JavaScript", "React", "Node.js", "Express.js", "MongoDB", "TypeScript", "Microservices", "Leadership"],
          salaryRange: "₹6,00,000 - ₹10,00,000",
          department: "Engineering",
          isActive: true,
          applicationDeadline: new Date(Date.now() + 40 * 24 * 60 * 60 * 1000)
        }
      ]);
      console.log('✅ Sample careers created:', careers.length);
    } else {
      console.log('ℹ️ Careers already exist, skipping career creation');
    }

    console.log('🎉 Database seeding completed successfully!');
    console.log('\n📊 Summary:');
    console.log(`- Admin user: ${adminUser.email}`);
    console.log(`- Blogs: ${blogs.length}`);
    console.log(`- Services: ${services.length}`);
    console.log(`- Team members: ${teamMembers.length}`);
    console.log(`- Testimonials: ${testimonials.length}`);
    console.log(`- Features: ${features.length}`);
    console.log(`- Notifications: ${notifications.length}`);
    console.log(`- Banners: ${banners.length}`);
    console.log(`- Company: ${company ? company.name : 'Already exists'}`);
    console.log(`- Careers: ${careers.length}`);

  } catch (error) {
    console.error('❌ Error seeding database:', error);
  } finally {
    await sequelize.close();
  }
};

// Run the seed function
seedDatabase();
