import { DataTypes, Model, type Optional, Op } from 'sequelize';
import sequelize from '../config/database.js';

export interface INotificationAttributes {
  id: number;
  notificationId: string;
  message: string;
  link: string | null;
  color: string;
  isActive: boolean;
  createdAt: Date;
  updatedAt: Date;
}

export interface INotificationCreationAttributes extends Optional<INotificationAttributes, 'id' | 'link' | 'color' | 'isActive' | 'createdAt' | 'updatedAt'> {}

export interface INotificationInstance extends Model<INotificationAttributes, INotificationCreationAttributes>, INotificationAttributes {}

const Notification = sequelize.define<INotificationInstance, INotificationCreationAttributes>('Notification', {
  id: {
    type: DataTypes.INTEGER,
    primaryKey: true,
    autoIncrement: true
  },
  notificationId: {
    type: DataTypes.STRING,
    allowNull: false,
    unique: true,
    validate: {
      notEmpty: true
    }
  },
  message: {
    type: DataTypes.STRING(200),
    allowNull: false,
    validate: {
      notEmpty: true,
      len: [1, 200]
    }
  },
  link: {
    type: DataTypes.STRING(500),
    allowNull: true,
    validate: {
      len: [0, 500]
    }
  },
  color: {
    type: DataTypes.STRING(7),
    allowNull: false,
    defaultValue: '#3B82F6',
    validate: {
      is: /^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/
    }
  },
  isActive: {
    type: DataTypes.BOOLEAN,
    defaultValue: false
  }
}, {
  tableName: 'notifications',
  timestamps: true,
  hooks: {
    beforeSave: async (notification: INotificationInstance) => {
      // Ensure only one notification can be active at a time
      if (notification.isActive) {
        await Notification.update(
          { isActive: false },
          { 
            where: { 
              isActive: true,
              id: { [Op.ne]: notification.id }
            }
          }
        );
      }
    }
  }
});

export default Notification;