import { DataTypes, Model, type Optional } from 'sequelize';
import sequelize from '../config/database.js';

export interface IJobApplicationAttributes {
  id: number;
  careerId: number;
  name: string;
  email: string;
  mobile: string;
  hasExperience: boolean;
  experienceDetails?: string;
  resume?: string;
  coverLetter?: string;
  status: 'pending' | 'reviewed' | 'shortlisted' | 'rejected' | 'hired';
  notes?: string;
  appliedAt: Date;
  reviewedAt?: Date;
  createdAt: Date;
  updatedAt: Date;
}

export interface IJobApplicationCreationAttributes extends Optional<IJobApplicationAttributes, 'id' | 'experienceDetails' | 'resume' | 'coverLetter' | 'notes' | 'reviewedAt' | 'createdAt' | 'updatedAt'> {}

export interface IJobApplicationInstance extends Model<IJobApplicationAttributes, IJobApplicationCreationAttributes>, IJobApplicationAttributes {}

const JobApplication = sequelize.define<IJobApplicationInstance, IJobApplicationCreationAttributes>('JobApplication', {
  id: {
    type: DataTypes.INTEGER,
    primaryKey: true,
    autoIncrement: true
  },
  careerId: {
    type: DataTypes.INTEGER,
    allowNull: false,
    references: {
      model: 'careers',
      key: 'id'
    },
    onDelete: 'CASCADE'
  },
  name: {
    type: DataTypes.STRING(100),
    allowNull: false,
    validate: {
      notEmpty: true,
      len: [1, 100]
    }
  },
  email: {
    type: DataTypes.STRING(255),
    allowNull: false,
    validate: {
      isEmail: true,
      notEmpty: true
    }
  },
  mobile: {
    type: DataTypes.STRING(15),
    allowNull: false,
    validate: {
      notEmpty: true,
      len: [10, 15]
    }
  },
  hasExperience: {
    type: DataTypes.BOOLEAN,
    allowNull: false,
    defaultValue: false
  },
  experienceDetails: {
    type: DataTypes.TEXT,
    allowNull: true
  },
  resume: {
    type: DataTypes.STRING(500),
    allowNull: true
  },
  coverLetter: {
    type: DataTypes.TEXT,
    allowNull: true
  },
  status: {
    type: DataTypes.ENUM('pending', 'reviewed', 'shortlisted', 'rejected', 'hired'),
    allowNull: false,
    defaultValue: 'pending'
  },
  notes: {
    type: DataTypes.TEXT,
    allowNull: true
  },
  appliedAt: {
    type: DataTypes.DATE,
    allowNull: false,
    defaultValue: DataTypes.NOW
  },
  reviewedAt: {
    type: DataTypes.DATE,
    allowNull: true
  }
}, {
  tableName: 'job_applications',
  timestamps: true
});

// Define associations
// Note: Associations will be defined after both models are loaded
// This will be handled in the main server file

export default JobApplication;
