import { DataTypes, Model, type Optional } from 'sequelize';
import sequelize from '../config/database.js';

export interface IContactAttributes {
  id: number;
  name: string;
  email: string | null;
  phone: string;
  serviceId: string;
  message: string;
  status: 'pending' | 'contacted' | 'resolved';
  createdAt: Date;
  updatedAt: Date;
}

export interface IContactCreationAttributes extends Optional<IContactAttributes, 'id' | 'email' | 'status' | 'createdAt' | 'updatedAt'> {}

export interface IContactInstance extends Model<IContactAttributes, IContactCreationAttributes>, IContactAttributes {}

const Contact = sequelize.define<IContactInstance, IContactCreationAttributes>('Contact', {
  id: {
    type: DataTypes.INTEGER,
    primaryKey: true,
    autoIncrement: true
  },
  name: {
    type: DataTypes.STRING(100),
    allowNull: false,
    validate: {
      notEmpty: true,
      len: [2, 100]
    }
  },
  email: {
    type: DataTypes.STRING,
    allowNull: true,
    validate: {
      isEmail: true
    }
  },
  phone: {
    type: DataTypes.STRING(20),
    allowNull: false,
    validate: {
      notEmpty: true
    }
  },
  serviceId: {
    type: DataTypes.STRING(50),
    allowNull: false,
    validate: {
      notEmpty: true
    }
  },
  message: {
    type: DataTypes.TEXT,
    allowNull: false,
    validate: {
      notEmpty: true,
      len: [10, 1000]
    }
  },
  status: {
    type: DataTypes.ENUM('pending', 'contacted', 'resolved'),
    defaultValue: 'pending'
  }
}, {
  tableName: 'contacts',
  timestamps: true
});

export default Contact;
