import { DataTypes, Model,type Optional } from 'sequelize';
import sequelize from '../config/database.js';

export interface ICompanyAttributes {
  id: number;
  legalName: string | null;
  displayName: string | null;
  tagline: string | null;
  about: string | null;
  name: string;
  email: string;
  phone: string;
  address: string;
  logo: string | null;
  darkLogo: string | null;
  // description: string | null;
  website: string | null;
  socialMedia: {
    facebook?: string;
    twitter?: string;
    linkedin?: string;
    instagram?: string;
    youtube?: string;
  } | null;
  founder: {
    name?: string;
    title?: string;
    email?: string;
    phone?: string;
    bio?: string;
    avatar?: string;
  } | null;
  createdAt: Date;
  updatedAt: Date;
}

export interface ICompanyCreationAttributes extends Optional<ICompanyAttributes, 'id' | 'legalName' | 'displayName' | 'tagline' | 'about' | 'logo' | 'darkLogo' |  'website' | 'socialMedia' | 'founder' | 'createdAt' | 'updatedAt'> {}

export interface ICompanyInstance extends Model<ICompanyAttributes, ICompanyCreationAttributes>, ICompanyAttributes {}

const Company = sequelize.define<ICompanyInstance, ICompanyCreationAttributes>('Company', {
  id: {
    type: DataTypes.INTEGER,
    primaryKey: true,
    autoIncrement: true
  },
  legalName: {
    type: DataTypes.STRING(100),
    allowNull: true
  },
  displayName: {
    type: DataTypes.STRING(100),
    allowNull: true,
     field: 'display_name'
  },
  tagline: {
    type: DataTypes.STRING(200),
    allowNull: true
  },
  about: {
    type: DataTypes.TEXT,
    allowNull: true
  },
  name: {
    type: DataTypes.STRING(100),
    allowNull: false,
    validate: {
      notEmpty: true,
      len: [1, 100]
    }
  },
  email: {
    type: DataTypes.STRING,
    allowNull: false,
    validate: {
      isEmail: true,
      notEmpty: true
    }
  },
  phone: {
    type: DataTypes.STRING(20),
    allowNull: false,
    validate: {
      notEmpty: true
    }
  },
  address: {
    type: DataTypes.STRING(500),
    allowNull: false,
    validate: {
      notEmpty: true
    }
  },
  logo: {
    type: DataTypes.STRING,
    allowNull: true
  },
  darkLogo: {
    type: DataTypes.STRING,
    allowNull: true,
      field: 'dark_logo'
  },
  // description: {
  //   type: DataTypes.TEXT,
  //   allowNull: true
  // },
  website: {
    type: DataTypes.STRING,
    allowNull: true,
    validate: {
      isUrl: true
    }
  },
  socialMedia: {
    type: DataTypes.JSON,
    allowNull: true,
    get() {
      const value = this.getDataValue('socialMedia');
      return value ? (typeof value === 'string' ? JSON.parse(value) : value) : null;
    },
    set(value: any) {
      this.setDataValue('socialMedia', value);
    },
    validate: {
      isValidSocialMedia(value: any) {
        if (value && typeof value === 'object') {
          const validKeys = ['facebook', 'twitter', 'linkedin', 'instagram', 'youtube'];
          const keys = Object.keys(value);
          if (keys.some(key => !validKeys.includes(key))) {
            throw new Error('Invalid social media key');
          }
        }
      }
    }
  },
  founder: {
    type: DataTypes.JSON,
    allowNull: true,
    get() {
      const value = this.getDataValue('founder');
      return value ? (typeof value === 'string' ? JSON.parse(value) : value) : null;
    },
    set(value: any) {
      this.setDataValue('founder', value);
    },
    validate: {
      isValidFounder(value: any) {
        if (value && typeof value === 'object') {
          const validKeys = ['name', 'title', 'email', 'phone', 'bio', 'avatar'];
          const keys = Object.keys(value);
          if (keys.some(key => !validKeys.includes(key))) {
            throw new Error('Invalid founder key');
          }
        }
      }
    }
  }
}, {
  tableName: 'companies',
  timestamps: true
});

export default Company;
