import  express from 'express';
import cors from 'cors';
import helmet from 'helmet';
import morgan from 'morgan';
import dotenv from 'dotenv';
import { createRequire } from 'module';
const require = createRequire(import.meta.url);
const passport = require('passport');
import { configurePassport } from './middleware/passport.js';
import authRoutes from './routes/auth.routes.js';
import featuresRoutes from './routes/features.routes.js';
import servicesRoutes from './routes/services.routes.js';
import testimonialsRoutes from './routes/testimonials.routes.js';
import teamRoutes from './routes/team.routes.js';
import blogsRoutes from './routes/blogs.routes.js';
import notificationRoutes from './routes/notification.routes.js';
import bannerRoutes from './routes/banner.routes.js';
import companyRoutes from './routes/company.routes.js';
import configRoutes from './routes/config.routes.js';
import otpRoutes from './routes/otp.routes.js';
import appointmentRoutes from './routes/appointment.routes.js';
import contactRoutes from './routes/contact.routes.js';
import whatWeDoRoutes from './routes/whatWeDo.routes.js';
import careerRoutes from './routes/career.routes.js';
import jobApplicationRoutes from './routes/jobApplication.routes.js';
import type { NextFunction, Request, Response } from 'express';
import path from 'node:path';
import sequelize from './config/database.js';

dotenv.config();

const app = express();
const PORT = process.env.PORT || 5252;

// Middleware
app.use(helmet({
  crossOriginResourcePolicy: { policy: 'cross-origin' },
}));
app.use(cors({
  origin: ["http://localhost:3000", "http://localhost:5252", "http://localhost:3001", "https://inficomsolutions.in"],
  credentials: true
}));
app.use(morgan('combined'));
app.use(express.json({ limit: '50mb' }));
app.use(express.urlencoded({ extended: true, limit: '50mb' }));
app.use('/uploads', express.static(path.join(process.cwd(), 'uploads')));
// Passport middleware
app.use(passport.initialize());
configurePassport(passport);

// Routes
app.use('/api/auth', authRoutes);
app.use('/api/features', featuresRoutes);
app.use('/api/testimonials', testimonialsRoutes);
app.use('/api/services', servicesRoutes);
app.use('/api/team', teamRoutes);
app.use('/api/blogs', blogsRoutes);
app.use('/api/notifications', notificationRoutes);
app.use('/api/banner', bannerRoutes);
app.use('/api/company', companyRoutes);
app.use('/api/config', configRoutes);
app.use('/api/otp', otpRoutes);
app.use('/api/appointments', appointmentRoutes);
app.use('/api/contacts', contactRoutes);
app.use('/api/what-we-do', whatWeDoRoutes);
app.use('/api/careers', careerRoutes);
app.use('/api/job-applications', jobApplicationRoutes);
// app.use('/api/upload', uploadRoutes);

// Serve uploaded files
app.use('/uploads', express.static('uploads'));

// Health check
app.get('/api/health', ( req: Request, res:Response) => {
  res.json({ status: 'OK', timestamp: new Date().toISOString() });
});
app.use('/api/uploads', express.static(
  path.join(process.cwd(), 'uploads'),
  {
    setHeaders: (res) => {
      res.setHeader('Cross-Origin-Resource-Policy', 'cross-origin');
      // Optional cache:
      // res.setHeader('Cache-Control', 'public, max-age=31536000, immutable');
    },
  }
));
// Global error handler
app.use((err: any, req: Request, res:Response, next: NextFunction) => {
  console.error(err.stack);
  res.status(500).json({ message: 'Something went wrong!' });
});

// Import all models to ensure they are registered
import './models/User.js';
import './models/Blog.js';
import './models/Service.js';
import './models/TeamMember.js';
import './models/Testimonial.js';
import './models/Feature.js';
import './models/Notification.js';
import './models/Banner.js';
import './models/Company.js';
import './models/Config.js';
import './models/Appointment.js';
import './models/Contact.js';
import './models/Career.js';
import './models/JobApplication.js';

// Define model associations after all models are loaded
import Career from './models/Career.js';
import JobApplication from './models/JobApplication.js';

// Define associations
Career.hasMany(JobApplication, {
  foreignKey: 'careerId',
  as: 'applications'
});

JobApplication.belongsTo(Career, {
  foreignKey: 'careerId',
  as: 'career'
});

// Connect to MySQL and sync models
const startServer = async () => {
  try {
    await sequelize.authenticate();
    console.log('Connected to MySQL database');
    
          // Sync all models with the database
          await sequelize.sync({ force: false });
    console.log('Database models synchronized');
    
    app.listen(PORT, () => {
      console.log(`Server is running on port ${PORT}`);
    });
  } catch (error) {
    console.error('Database connection error:', error);
    process.exit(1);
  }
};

startServer();