import type { Request, Response } from "express";
import Notification from "../models/Notification.js";
// import Notification, { INotification } from '../models/Notification';

export const getAllNotifications = async (req: Request, res: Response): Promise<void> => {
  try {
    const notifications = await Notification.findAll({ 
      order: [['createdAt', 'DESC']] 
    });
    res.json(notifications);
  } catch (error: any) {
    console.error('Get notifications error:', error);
    res.status(500).json({ message: error.message || 'Server error' });
  }
};

export const getActiveNotification = async (req: Request, res: Response): Promise<void> => {
  try {
    const notification = await Notification.findOne({ where: { isActive: true } });
    res.json(notification);
  } catch (error: any) {
    console.error('Get active notification error:', error);
    res.status(500).json({ message: error.message || 'Server error' });
  }
};

export const getNotificationById = async (req: Request, res: Response): Promise<void> => {
  try {
    const notification = await Notification.findByPk(req.params.id);
    if (!notification) {
      res.status(404).json({ message: 'Notification not found' });
      return;
    }
    res.json(notification);
  } catch (error: any) {
    console.error('Get notification error:', error);
    res.status(500).json({ message: error.message || 'Server error' });
  }
};

export const createNotification = async (req: Request, res: Response): Promise<void> => {
  try {
    const { id, message, link, color, isActive } = req.body;

    // If creating an active notification, deactivate all others
    if (isActive) {
      await Notification.update({ isActive: false }, { where: {} });
    }

    const notification = await Notification.create({
      notificationId: id,
      message,
      link,
      color,
      isActive
    });
    res.status(201).json(notification);
  } catch (error: any) {
    console.error('Create notification error:', error);
    res.status(400).json({ message: error.message || 'Server error' });
  }
};

export const updateNotification = async (req: Request, res: Response): Promise<void> => {
  try {
    const { message, link, color, isActive } = req.body;

    // If activating this notification, deactivate all others
    if (isActive) {
      await Notification.update(
        { isActive: false }, 
        { where: { id: { [require('sequelize').Op.ne]: req.params.id } } }
      );
    }

    const [updatedRows] = await Notification.update(
      { message, link, color, isActive },
      { 
        where: { id: req.params.id },
        returning: true
      }
    );

    if (updatedRows === 0) {
      res.status(404).json({ message: 'Notification not found' });
      return;
    }

    const notification = await Notification.findByPk(req.params.id);
    res.json(notification);
  } catch (error: any) {
    console.error('Update notification error:', error);
    res.status(400).json({ message: error.message || 'Server error' });
  }
};

export const deleteNotification = async (req: Request, res: Response): Promise<void> => {
  try {
    const deletedRows = await Notification.destroy({
      where: { id: req.params.id }
    });
    
    if (deletedRows === 0) {
      res.status(404).json({ message: 'Notification not found' });
      return;
    }
    
    res.json({ message: 'Notification deleted successfully' });
  } catch (error: any) {
    console.error('Delete notification error:', error);
    res.status(500).json({ message: error.message || 'Server error' });
  }
};