import type { Request, Response } from 'express';
import JobApplication from '../models/JobApplication.js';
import Career from '../models/Career.js';
import { documentUpload, toPublicUrl } from '../middleware/documentUpload.js';
import { tempStorage } from '../services/tempStorage.js';
// // import { sendOTP } from '../utils/otpService.js';

// Get all job applications with filters
export const getAllJobApplications = async (req: Request, res: Response) => {
  try {
    const { status, careerId, page = 1, limit = 10 } = req.query;
    
    const whereClause: any = {};
    if (status) whereClause.status = status;
    if (careerId) whereClause.careerId = careerId;

    const offset = (Number(page) - 1) * Number(limit);

    const { count, rows: applications } = await JobApplication.findAndCountAll({
      where: whereClause,
      include: [{
        model: Career,
        as: 'career',
        attributes: ['id', 'title', 'department', 'location']
      }],
      order: [['appliedAt', 'DESC']],
      limit: Number(limit),
      offset
    });

    res.json({
      applications,
      pagination: {
        total: count,
        page: Number(page),
        limit: Number(limit),
        pages: Math.ceil(count / Number(limit))
      }
    });
  } catch (error) {
    console.error('Error fetching job applications:', error);
    res.status(500).json({ message: 'Error fetching job applications' });
  }
};

// Get single job application
export const getJobApplicationById = async (req: Request, res: Response) => {
  try {
    const { id } = req.params;
    
    const application = await JobApplication.findByPk(id, {
      include: [{
        model: Career,
        as: 'career',
        attributes: ['id', 'title', 'department', 'location', 'description']
      }]
    });

    if (!application) {
      return res.status(404).json({ message: 'Job application not found' });
    }

    res.json(application);
  } catch (error) {
    console.error('Error fetching job application:', error);
    res.status(500).json({ message: 'Error fetching job application' });
  }
};

// Store job application data temporarily (OTP will be sent by frontend using OTP controller)
export const storeJobApplicationData = async (req: Request, res: Response) => {
  try {
    const { careerId, name, email, mobile, hasExperience, experienceDetails, coverLetter } = req.body;

    // Validate required fields
    if (!careerId || !name || !email || !mobile) {
      return res.status(400).json({ message: 'Missing required fields' });
    }

    // Check if career exists
    const career = await Career.findByPk(careerId);
    if (!career) {
      return res.status(404).json({ message: 'Job opening not found' });
    }

    // Check if user already applied for this job
    const existingApplication = await JobApplication.findOne({
      where: { careerId, email }
    });

    if (existingApplication) {
      return res.status(400).json({ message: 'You have already applied for this position' });
    }

    // Handle file uploads if present
    let resumePath = null;
    let coverLetterPath = null;

    if (req.files) {
      const files = req.files as { [fieldname: string]: Express.Multer.File[] };
      
      // Find resume file
      if (files.resume && files.resume.length > 0 && files.resume[0]) {
        resumePath = toPublicUrl(files.resume[0].filename);
      }

      // Find cover letter file
      if (files.coverLetter && files.coverLetter.length > 0 && files.coverLetter[0]) {
        coverLetterPath = toPublicUrl(files.coverLetter[0].filename);
      }
    }

    // Store application data temporarily
    const tempApplicationData = {
      careerId,
      name,
      email,
      mobile,
      hasExperience: hasExperience || false,
      experienceDetails,
      coverLetter: coverLetter || coverLetterPath,
      resume: resumePath,
      timestamp: Date.now()
    };

    // Store in temporary storage (not database)
    tempStorage.store(mobile, tempApplicationData);

    res.json({
      message: 'Application data stored temporarily. Please verify your mobile number to complete the application.',
      success: true
    });
  } catch (error) {
    console.error('Error storing job application data:', error);
    res.status(500).json({ message: 'Error storing application data' });
  }
};

// Create job application (OTP verification should be done by frontend using OTP controller first)
export const createJobApplication = async (req: Request, res: Response) => {
  try {
    const { mobile } = req.body;

    if (!mobile) {
      return res.status(400).json({ message: 'Mobile number is required' });
    }

    // Get temporary application data from temporary storage
    const tempApplicationData = tempStorage.get(mobile);
    
    if (!tempApplicationData) {
      return res.status(400).json({ message: 'No pending application found for this mobile number' });
    }

    // Check if data is still valid (not expired)
    if (!tempStorage.isValid(mobile, 30)) {
      tempStorage.remove(mobile);
      return res.status(400).json({ message: 'Application data has expired. Please start over.' });
    }

    // Create the application
    const applicationData: any = {
      careerId: tempApplicationData.careerId,
      name: tempApplicationData.name,
      email: tempApplicationData.email,
      mobile: tempApplicationData.mobile,
      hasExperience: tempApplicationData.hasExperience,
      experienceDetails: tempApplicationData.experienceDetails,
      status: 'pending',
      appliedAt: new Date()
    };

    // Add optional fields only if they exist
    if (tempApplicationData.coverLetter) {
      applicationData.coverLetter = tempApplicationData.coverLetter;
    }
    if (tempApplicationData.resume) {
      applicationData.resume = tempApplicationData.resume;
    }

    const application = await JobApplication.create(applicationData);

    // Clean up temporary data from temporary storage
    tempStorage.remove(mobile);

    res.json({
      message: 'Application submitted successfully!',
      application
    });
  } catch (error) {
    console.error('Error creating job application:', error);
    res.status(500).json({ message: 'Error creating job application' });
  }
};

// Update job application status
export const updateJobApplicationStatus = async (req: Request, res: Response) => {
  try {
    const { id } = req.params;
    const { status, notes } = req.body;

    const application = await JobApplication.findByPk(id);
    
    if (!application) {
      return res.status(404).json({ message: 'Job application not found' });
    }

    const updateData: any = { status };
    if (notes) updateData.notes = notes;
    if (status !== 'pending') updateData.reviewedAt = new Date();

    await application.update(updateData);

    res.json(application);
  } catch (error) {
    console.error('Error updating job application:', error);
    res.status(500).json({ message: 'Error updating job application' });
  }
};

// Delete job application
export const deleteJobApplication = async (req: Request, res: Response) => {
  try {
    const { id } = req.params;
    
    const application = await JobApplication.findByPk(id);
    
    if (!application) {
      return res.status(404).json({ message: 'Job application not found' });
    }

    await application.destroy();

    res.json({ message: 'Job application deleted successfully' });
  } catch (error) {
    console.error('Error deleting job application:', error);
    res.status(500).json({ message: 'Error deleting job application' });
  }
};

// Get job application statistics
export const getJobApplicationStats = async (req: Request, res: Response) => {
  try {
    const totalApplications = await JobApplication.count();
    const pendingApplications = await JobApplication.count({ where: { status: 'pending' } });
    const reviewedApplications = await JobApplication.count({ where: { status: 'reviewed' } });
    const shortlistedApplications = await JobApplication.count({ where: { status: 'shortlisted' } });
    const rejectedApplications = await JobApplication.count({ where: { status: 'rejected' } });
    const hiredApplications = await JobApplication.count({ where: { status: 'hired' } });

    const applicationsByCareer = await JobApplication.findAll({
      attributes: [
        'careerId',
        [JobApplication.sequelize!.fn('COUNT', JobApplication.sequelize!.col('JobApplication.id')), 'count']
      ],
      include: [{
        model: Career,
        as: 'career',
        attributes: ['title', 'department']
      }],
      group: ['JobApplication.careerId', 'career.id'],
      raw: true
    });

    res.json({
      total: totalApplications,
      pending: pendingApplications,
      reviewed: reviewedApplications,
      shortlisted: shortlistedApplications,
      rejected: rejectedApplications,
      hired: hiredApplications,
      byCareer: applicationsByCareer
    });
  } catch (error) {
    console.error('Error fetching job application stats:', error);
    res.status(500).json({ message: 'Error fetching job application statistics' });
  }
};

