import type { Request, Response } from "express";
import Feature from "../models/Feature.js";


export const getAllFeatures = async (req: Request, res: Response): Promise<void> => {
  try {
    const features = await Feature.findAll({ 
      order: [['createdAt', 'DESC']] 
    });
    res.json(features);
  } catch (error: any) {
    console.error('Get features error:', error);
    res.status(500).json({ message: error.message || 'Server error' });
  }
};

export const getFeatureById = async (req: Request, res: Response): Promise<void> => {
  try {
    const feature = await Feature.findByPk(req.params.id);
    if (!feature) {
      res.status(404).json({ message: 'Feature not found' });
      return;
    }
    res.json(feature);
  } catch (error: any) {
    console.error('Get feature error:', error);
    res.status(500).json({ message: error.message || 'Server error' });
  }
};

export const createFeature = async (req: Request, res: Response): Promise<void> => {
  try {
    const { title, desc, iconKey, color } = req.body;

    const feature = await Feature.create({
      title,
      desc,
      iconKey,
      color
    });
    res.status(201).json(feature);
  } catch (error: any) {
    console.error('Create feature error:', error);
    res.status(400).json({ message: error.message || 'Server error' });
  }
};

export const updateFeature = async (req: Request, res: Response): Promise<void> => {
  try {
    const { title, desc, iconKey, color } = req.body;

    const [updatedRows] = await Feature.update(
      { title, desc, iconKey, color },
      { 
        where: { id: req.params.id },
        returning: true
      }
    );

    if (updatedRows === 0) {
      res.status(404).json({ message: 'Feature not found' });
      return;
    }

    const feature = await Feature.findByPk(req.params.id);
    res.json(feature);
  } catch (error: any) {
    console.error('Update feature error:', error);
    res.status(400).json({ message: error.message || 'Server error' });
  }
};

export const deleteFeature = async (req: Request, res: Response): Promise<void> => {
  try {
    const deletedRows = await Feature.destroy({
      where: { id: req.params.id }
    });
    
    if (deletedRows === 0) {
      res.status(404).json({ message: 'Feature not found' });
      return;
    }
    
    res.json({ message: 'Feature deleted successfully' });
  } catch (error: any) {
    console.error('Delete feature error:', error);
    res.status(500).json({ message: error.message || 'Server error' });
  }
};