import type { Request, Response } from 'express';
import Contact from '../models/Contact.js';

export const createContact = async (req: Request, res: Response): Promise<void> => {
  try {
    const { name, email, phone, serviceId, message } = req.body;

    if (!name || !phone || !serviceId || !message) {
      res.status(400).json({ message: 'Name, phone, service, and message are required' });
      return;
    }

    const contact = await Contact.create({
      name,
      email: email || null,
      phone,
      serviceId,
      message,
      status: 'pending'
    });

    res.status(201).json(contact);
  } catch (error: any) {
    console.error('Create contact error:', error);
    res.status(400).json({ message: error.message || 'Server error' });
  }
};

export const getAllContacts = async (req: Request, res: Response): Promise<void> => {
  try {
    const contacts = await Contact.findAll({
      order: [['createdAt', 'DESC']]
    });

    res.json(contacts);
  } catch (error: any) {
    console.error('Get contacts error:', error);
    res.status(500).json({ message: error.message || 'Server error' });
  }
};

export const getContactById = async (req: Request, res: Response): Promise<void> => {
  try {
    const { id } = req.params;
    const contact = await Contact.findByPk(id);

    if (!contact) {
      res.status(404).json({ message: 'Contact not found' });
      return;
    }

    res.json(contact);
  } catch (error: any) {
    console.error('Get contact error:', error);
    res.status(500).json({ message: error.message || 'Server error' });
  }
};

export const updateContact = async (req: Request, res: Response): Promise<void> => {
  try {
    const { id } = req.params;
    const { status } = req.body;

    const contact = await Contact.findByPk(id);

    if (!contact) {
      res.status(404).json({ message: 'Contact not found' });
      return;
    }

    await contact.update({ status });

    res.json(contact);
  } catch (error: any) {
    console.error('Update contact error:', error);
    res.status(400).json({ message: error.message || 'Server error' });
  }
};

export const deleteContact = async (req: Request, res: Response): Promise<void> => {
  try {
    const { id } = req.params;

    const contact = await Contact.findByPk(id);

    if (!contact) {
      res.status(404).json({ message: 'Contact not found' });
      return;
    }

    await contact.destroy();

    res.json({ message: 'Contact deleted successfully' });
  } catch (error: any) {
    console.error('Delete contact error:', error);
    res.status(500).json({ message: error.message || 'Server error' });
  }
};
