import type { Request, Response } from 'express';
import Config from '../models/Config.js';

// Configuration registry with all expected keys
const CONFIG_REGISTRY = [
  { key: 'APP_NAME', label: 'App Name', required: true, secret: false },
  { key: 'FRONTEND_URL', label: 'Frontend URL', required: true, secret: false },
  { key: 'FROM_EMAIL', label: 'Mailer From Email', required: true, secret: false },
  { key: 'FROM_PASS', label: 'Mailer App Password', required: true, secret: true },
  { key: 'SMTP_HOST', label: 'SMTP Host', required: true, secret: false },
  { key: 'SMTP_PORT', label: 'SMTP Port', required: true, secret: false },
  { key: 'SMTP_SECURE', label: 'SMTP Secure', required: false, secret: false },
  { key: 'SMTP_USER', label: 'SMTP Username', required: true, secret: false },
  { key: 'SMTP_PASS', label: 'SMTP Password', required: true, secret: true },
  { key: 'JWT_SECRET', label: 'JWT Secret', required: true, secret: true },
  { key: 'WHATSAPP_INSTANCE_ID', label: 'WhatsApp Instance ID', required: false, secret: false },
  { key: 'WHATSAPP_ACCESS_TOKEN', label: 'WhatsApp Access Token', required: false, secret: true },
];

export const getConfig = async (req: Request, res: Response): Promise<void> => {
  try {
    // Get all existing configs from database
    const existingConfigs = await Config.findAll();
    const configMap = new Map(existingConfigs.map(config => [config.key, config]));

    // Build response with all registry items
    const items = CONFIG_REGISTRY.map(registryItem => {
      const existing = configMap.get(registryItem.key);
      
      return {
        key: registryItem.key,
        valuePreview: existing ? (registryItem.secret ? '***' : existing.value) : null,
        exists: !!existing,
        secret: registryItem.secret,
        required: registryItem.required,
        updatedAt: existing?.updatedAt?.toISOString(),
        label: registryItem.label
      };
    });

    res.json({ items });
  } catch (error: any) {
    console.error('Get config error:', error);
    res.status(500).json({ message: error.message || 'Server error' });
  }
};

export const getConfigStats = async (req: Request, res: Response): Promise<void> => {
  try {
    const totalKeysInRegistry = CONFIG_REGISTRY.length;
    const existingConfigs = await Config.findAll();
    const keysPresentInDB = existingConfigs.length;
    const keysMissingInDB = totalKeysInRegistry - keysPresentInDB;
    
    const requiredMissing = CONFIG_REGISTRY
      .filter(item => item.required)
      .filter(item => !existingConfigs.some(config => config.key === item.key))
      .map(item => item.key);
    
    const healthy = requiredMissing.length === 0;

    res.json({
      totalKeysInRegistry,
      keysPresentInDB,
      keysMissingInDB,
      requiredMissing,
      healthy
    });
  } catch (error: any) {
    console.error('Get config stats error:', error);
    res.status(500).json({ message: error.message || 'Server error' });
  }
};

export const upsertConfig = async (req: Request, res: Response): Promise<void> => {
  try {
    const { key, value, secret } = req.body;

    if (!key || !value) {
      res.status(400).json({ message: 'Key and value are required' });
      return;
    }

    // Check if key exists in registry
    const registryItem = CONFIG_REGISTRY.find(item => item.key === key);
    if (!registryItem) {
      res.status(400).json({ message: 'Invalid configuration key' });
      return;
    }

    // Find existing config or create new one
    const [config, created] = await Config.findOrCreate({
      where: { key },
      defaults: {
        key,
        value,
        secret: secret || registryItem.secret,
        required: registryItem.required,
        label: registryItem.label
      }
    });

    // If not created, update the existing one
    if (!created) {
      await config.update({
        value,
        secret: secret !== undefined ? secret : config.secret
      });
    }

    res.json({ ok: true });
  } catch (error: any) {
    console.error('Upsert config error:', error);
    res.status(400).json({ message: error.message || 'Server error' });
  }
};
