import type { Request, Response } from 'express';
import Company from '../models/Company.js';
import { toPublicUrl } from '../middleware/upload.js';

export const getCompany = async (req: Request, res: Response): Promise<void> => {
  try {
    // Get the first company (assuming single company setup)
    const company = await Company.findOne({ order: [['createdAt', 'ASC']] });
    
    console.log("company",company);
    
    res.json(company);
  } catch (error: any) {
    console.error('Get company error:', error);
    res.status(500).json({ message: error.message || 'Server error' });
  }
};

export const createCompany = async (req: Request, res: Response): Promise<void> => {
  try {
    const { 
      legalName, 
      displayName, 
      tagline, 
      about, 
      website, 
      email, 
      phone, 
      address,
      'social.facebook': socialFacebook,
      'social.twitter': socialTwitter,
      'social.linkedin': socialLinkedin,
      'social.instagram': socialInstagram,
      'social.youtube': socialYoutube,
      'founder.name': founderName,
      'founder.title': founderTitle,
      'founder.email': founderEmail,
      'founder.phone': founderPhone,
      'founder.bio': founderBio
    } = req.body;

    // Check if company already exists
    const existingCompany = await Company.findOne();
    if (existingCompany) {
      res.status(400).json({ message: 'Company already exists. Use PUT to update.' });
      return;
    }

    // Handle file uploads
    const files = req.files as { [fieldname: string]: Express.Multer.File[] } | undefined;
    const logoFile = files?.logoFile?.[0];
    const darkLogoFile = files?.darkLogoFile?.[0];
    const founderAvatarFile = files?.founderAvatarFile?.[0];

    // Build social media object
    const socialMedia = {
      facebook: socialFacebook || null,
      twitter: socialTwitter || null,
      linkedin: socialLinkedin || null,
      instagram: socialInstagram || null,
      youtube: socialYoutube || null
    };

    // Build founder object
    const founder: {
      name?: string;
      title?: string;
      email?: string;
      phone?: string;
      bio?: string;
      avatar?: string;
    } = {
      ...(founderName && { name: founderName }),
      ...(founderTitle && { title: founderTitle }),
      ...(founderEmail && { email: founderEmail }),
      ...(founderPhone && { phone: founderPhone }),
      ...(founderBio && { bio: founderBio }),
      ...(founderAvatarFile && { avatar: toPublicUrl(founderAvatarFile.filename) })
    };

    const company = await Company.create({
      legalName,
      displayName,
      tagline,
      about,
      name: displayName || legalName || 'Company',
      email,
      phone,
      address,
      description: about,
      website,
      logo: logoFile ? toPublicUrl(logoFile.filename) : null,
      darkLogo: darkLogoFile ? toPublicUrl(darkLogoFile.filename) : null,
      socialMedia: Object.values(socialMedia).some(v => v) ? socialMedia : null,
      founder: Object.values(founder).some(v => v) ? founder : null
    });

    res.status(201).json(company);
  } catch (error: any) {
    console.error('Create company error:', error);
    res.status(400).json({ message: error.message || 'Server error' });
  }
};

export const updateCompany = async (req: Request, res: Response): Promise<void> => {
  try {
    const { 
      legalName, 
      displayName, 
      tagline, 
      about, 
      website, 
      email, 
      phone, 
      address,
      logo,
      darkLogo,
      'social.facebook': socialFacebook,
      'social.twitter': socialTwitter,
      'social.linkedin': socialLinkedin,
      'social.instagram': socialInstagram,
      'social.youtube': socialYoutube,
      'founder.name': founderName,
      'founder.title': founderTitle,
      'founder.email': founderEmail,
      'founder.phone': founderPhone,
      'founder.bio': founderBio,
      'founder.avatar': founderAvatar
    } = req.body;

    // Get the first company (assuming single company setup)
    const company = await Company.findOne({ order: [['createdAt', 'ASC']] });
    
    if (!company) {
      res.status(404).json({ message: 'Company not found' });
      return;
    }

    // Handle file uploads
    const files = req.files as { [fieldname: string]: Express.Multer.File[] } | undefined;
    const logoFile = files?.logoFile?.[0];
    const darkLogoFile = files?.darkLogoFile?.[0];
    const founderAvatarFile = files?.founderAvatarFile?.[0];

    // Build social media object
    const socialMedia = {
      facebook: socialFacebook || null,
      twitter: socialTwitter || null,
      linkedin: socialLinkedin || null,
      instagram: socialInstagram || null,
      youtube: socialYoutube || null
    };

    // Build founder object
    const founder: {
      name?: string;
      title?: string;
      email?: string;
      phone?: string;
      bio?: string;
      avatar?: string;
    } = {
      ...(founderName && { name: founderName }),
      ...(founderTitle && { title: founderTitle }),
      ...(founderEmail && { email: founderEmail }),
      ...(founderPhone && { phone: founderPhone }),
      ...(founderBio && { bio: founderBio }),
      ...(founderAvatarFile && { avatar: toPublicUrl(founderAvatarFile.filename) }),
      ...(founderAvatar && !founderAvatarFile && { avatar: founderAvatar })
    };

    await company.update({
      legalName: legalName || company.legalName,
      displayName: displayName || company.displayName,
      tagline: tagline || company.tagline,
      about: about || company.about,
      name: displayName || legalName || company.name,
      email,
      phone,
      address,
      description: about || company.description,
      website,
      logo: logoFile ? toPublicUrl(logoFile.filename) : logo || company.logo,
      darkLogo: darkLogoFile ? toPublicUrl(darkLogoFile.filename) : darkLogo || company.darkLogo,
      socialMedia: Object.values(socialMedia).some(v => v) ? socialMedia : company.socialMedia,
      founder: Object.values(founder).some(v => v) ? founder : company.founder
    });

    res.json(company);
  } catch (error: any) {
    console.error('Update company error:', error);
    res.status(400).json({ message: error.message || 'Server error' });
  }
};
