import type { Request, Response } from 'express';
import Banner from '../models/Banner.js';

export const getBanner = async (req: Request, res: Response): Promise<void> => {
  try {
    const banner = await Banner.findOne({ where: { isActive: true } });
    res.json(banner);
  } catch (error: any) {
    console.error('Get banner error:', error);
    res.status(500).json({ message: error.message || 'Server error' });
  }
};

export const createBanner = async (req: Request, res: Response): Promise<void> => {
  try {
    const { message, link, color } = req.body;

    // If creating an active banner, deactivate all others
    if (req.body.isActive) {
      await Banner.update({ isActive: false }, { where: {} });
    }

    const banner = await Banner.create({
      message,
      link,
      color: color || '#3B82F6',
      isActive: req.body.isActive || false
    });

    res.status(201).json(banner);
  } catch (error: any) {
    console.error('Create banner error:', error);
    res.status(400).json({ message: error.message || 'Server error' });
  }
};

export const updateBanner = async (req: Request, res: Response): Promise<void> => {
  try {
    const { message, link, color, isActive } = req.body;

    // If activating this banner, deactivate all others
    if (isActive) {
      await Banner.update(
        { isActive: false }, 
        { where: { id: { [require('sequelize').Op.ne]: req.params.id } } }
      );
    }

    const [updatedRows] = await Banner.update(
      { message, link, color, isActive },
      { 
        where: { id: req.params.id },
        returning: true
      }
    );

    if (updatedRows === 0) {
      res.status(404).json({ message: 'Banner not found' });
      return;
    }

    const banner = await Banner.findByPk(req.params.id);
    res.json(banner);
  } catch (error: any) {
    console.error('Update banner error:', error);
    res.status(400).json({ message: error.message || 'Server error' });
  }
};

export const deleteBanner = async (req: Request, res: Response): Promise<void> => {
  try {
    const deletedRows = await Banner.destroy({
      where: { id: req.params.id }
    });
    
    if (deletedRows === 0) {
      res.status(404).json({ message: 'Banner not found' });
      return;
    }
    
    res.json({ message: 'Banner deleted successfully' });
  } catch (error: any) {
    console.error('Delete banner error:', error);
    res.status(500).json({ message: error.message || 'Server error' });
  }
};
