import type { Request, Response } from 'express';
import Appointment from '../models/Appointment.js';

export const createAppointment = async (req: Request, res: Response): Promise<void> => {
  try {
    const { name, email, phone, service, date, time, message } = req.body;

    if (!name || !email || !phone || !service || !date || !time || !message) {
      res.status(400).json({ message: 'All fields are required' });
      return;
    }

    const appointment = await Appointment.create({
      name,
      email,
      phone,
      service,
      date,
      time,
      message,
      status: 'pending'
    });

    res.status(201).json(appointment);
  } catch (error: any) {
    console.error('Create appointment error:', error);
    res.status(400).json({ message: error.message || 'Server error' });
  }
};

export const getAllAppointments = async (req: Request, res: Response): Promise<void> => {
  try {
    const appointments = await Appointment.findAll({
      order: [['createdAt', 'DESC']]
    });

    res.json(appointments);
  } catch (error: any) {
    console.error('Get appointments error:', error);
    res.status(500).json({ message: error.message || 'Server error' });
  }
};

export const getAppointmentById = async (req: Request, res: Response): Promise<void> => {
  try {
    const { id } = req.params;
    const appointment = await Appointment.findByPk(id);

    if (!appointment) {
      res.status(404).json({ message: 'Appointment not found' });
      return;
    }

    res.json(appointment);
  } catch (error: any) {
    console.error('Get appointment error:', error);
    res.status(500).json({ message: error.message || 'Server error' });
  }
};

export const updateAppointment = async (req: Request, res: Response): Promise<void> => {
  try {
    const { id } = req.params;
    const { status } = req.body;

    const appointment = await Appointment.findByPk(id);

    if (!appointment) {
      res.status(404).json({ message: 'Appointment not found' });
      return;
    }

    await appointment.update({ status });

    res.json(appointment);
  } catch (error: any) {
    console.error('Update appointment error:', error);
    res.status(400).json({ message: error.message || 'Server error' });
  }
};

export const deleteAppointment = async (req: Request, res: Response): Promise<void> => {
  try {
    const { id } = req.params;

    const appointment = await Appointment.findByPk(id);

    if (!appointment) {
      res.status(404).json({ message: 'Appointment not found' });
      return;
    }

    await appointment.destroy();

    res.json({ message: 'Appointment deleted successfully' });
  } catch (error: any) {
    console.error('Delete appointment error:', error);
    res.status(500).json({ message: error.message || 'Server error' });
  }
};
