# MongoDB to MySQL Migration Guide

This guide documents the migration from MongoDB (Mongoose) to MySQL (Sequelize) for the Inficom Solutions server.

## Changes Made

### 1. Dependencies
- **Removed**: `mongoose`, `@types/mongoose`
- **Added**: `sequelize`, `mysql2`, `@types/sequelize`

### 2. Database Configuration
- Created `config/database.ts` with Sequelize configuration
- Updated `index.ts` to use Sequelize instead of Mongoose
- Added model imports and database synchronization

### 3. Models Conversion
All Mongoose models were converted to Sequelize models:

#### User Model (`models/User.ts`)
- Converted to Sequelize with proper TypeScript interfaces
- Maintained password hashing with bcrypt
- Added instance method for password comparison
- Changed `_id` to `id` (integer primary key)

#### Blog Model (`models/Blog.ts`)
- Converted to Sequelize with validation
- Maintained all field validations and constraints
- Changed `_id` to `id` (integer primary key)

#### Service Model (`models/Service.ts`)
- Converted to Sequelize with JSON fields for arrays
- Maintained complex validation for nested objects
- Changed `_id` to `id` (integer primary key)

#### Other Models
- `TeamMember.ts` - Converted to Sequelize
- `Testimonial.ts` - Converted to Sequelize
- `Feature.ts` - Converted to Sequelize
- `Notification.ts` - Converted to Sequelize

### 4. Controllers Updates
All controllers were updated to use Sequelize methods:

#### Auth Controller (`controllers/auth.controller.ts`)
- `User.findOne({ email })` → `User.findOne({ where: { email } })`
- `User.findById(id)` → `User.findByPk(id)`
- `user.save()` → `user.update()`
- `new User(data)` → `User.create(data)`

#### Blog Controller (`controllers/blogs.controller.ts`)
- `Blog.find().sort()` → `Blog.findAll({ order: [['createdAt', 'DESC']] })`
- `Blog.findById(id)` → `Blog.findByPk(id)`
- `Blog.findByIdAndUpdate()` → `Blog.update()`
- `Blog.findByIdAndDelete()` → `Blog.destroy()`

#### Service Controller (`controllers/services.controller.ts`)
- Similar pattern updates as Blog controller

### 5. Middleware Updates
#### Auth Middleware (`middleware/auth.ts`)
- `User.findById().select('-password')` → `User.findByPk(id, { attributes: { exclude: ['password'] } })`
- `userDoc.toObject()` → `userDoc.toJSON()`

#### Passport Middleware (`middleware/passport.ts`)
- Updated JWT and Local strategies to use Sequelize methods

### 6. Type Definitions
- Updated `types/express.d.ts` to use `IUserInstance` instead of `IUser`

## Environment Variables

Create a `.env` file with the following variables:

```env
# Server Configuration
PORT=5000
NODE_ENV=development

# JWT Configuration
JWT_SECRET=your-super-secret-jwt-key-here

# MySQL Database Configuration
DB_HOST=localhost
DB_PORT=3306
DB_NAME=inficom_admin
DB_USER=root
DB_PASSWORD=your-mysql-password

# Email Configuration (for password reset)
FROM_EMAIL=your-email@gmail.com
FROM_PASS=your-app-password
```

## Database Setup

1. Install MySQL server
2. Create database: `CREATE DATABASE inficom_admin;`
3. Update `.env` with your MySQL credentials
4. Run the server - Sequelize will automatically create tables

## Key Differences

### MongoDB vs MySQL
- **IDs**: MongoDB uses `_id` (ObjectId), MySQL uses `id` (integer)
- **Queries**: MongoDB uses `find()`, MySQL uses `findAll()`, `findByPk()`
- **Updates**: MongoDB uses `findByIdAndUpdate()`, MySQL uses `update()`
- **Deletes**: MongoDB uses `findByIdAndDelete()`, MySQL uses `destroy()`

### Sequelize Specific
- **Associations**: Can be added later for relationships
- **Migrations**: Can be added for version control
- **Transactions**: Built-in support for database transactions
- **Validation**: Model-level validation with custom validators

## Testing

1. Start MySQL server
2. Create database
3. Update `.env` file
4. Run `npm run dev`
5. Test all API endpoints

## Rollback Plan

If rollback is needed:
1. Restore original Mongoose models
2. Reinstall `mongoose` and `@types/mongoose`
3. Update `index.ts` to use MongoDB connection
4. Update all controllers to use Mongoose methods

## Notes

- All existing API endpoints remain the same
- File upload functionality is unchanged
- Authentication flow is unchanged
- All validations and business logic are preserved
